"""ToolSet 客户端 / ToolSet Client

此模块提供工具集的客户端API。
This module provides the client API for toolsets.
"""

from typing import Optional

from alibabacloud_devs20230714.models import ListToolsetsRequest

from agentrun.toolset.api.control import ToolControlAPI
from agentrun.toolset.model import ToolSetListInput
from agentrun.utils.config import Config
from agentrun.utils.exception import HTTPError

from .toolset import ToolSet


class ToolSetClient:
    """ToolSet 客户端 / ToolSet Client

    提供工具集的获取和列表功能。
    Provides get and list functions for toolsets.
    """

    def __init__(self, config: Optional[Config] = None):
        """初始化客户端 / Initialize client

        Args:
            config: 配置对象,可选 / Configuration object, optional
        """
        self.__control_api = ToolControlAPI(config)

    async def get_async(
        self,
        name: str,
        config: Optional[Config] = None,
    ):
        try:
            result = await self.__control_api.get_toolset_async(
                name=name,
                config=config,
            )
        except HTTPError as e:
            raise e.to_resource_error("ToolSet", name) from e

        return ToolSet.from_inner_object(result)

    async def list_async(
        self,
        input: Optional[ToolSetListInput] = None,
        config: Optional[Config] = None,
    ):
        if input is None:
            input = ToolSetListInput()

        result = await self.__control_api.list_toolsets_async(
            input=ListToolsetsRequest().from_map(input.model_dump()),
            config=config,
        )

        return [ToolSet.from_inner_object(item) for item in result.data]
